import argparse, hashlib, json, requests, time, uuid


class YouDaoLLMTranslator:
    def __init__(self, app_key, app_secret):
        """
        初始化有道大模型翻译客户端

        :param app_key: 应用ID
        :param app_secret: 应用密钥
        """
        self.app_key = app_key
        self.app_secret = app_secret
        self.api_url = "https://openapi.youdao.com/proxy/http/llm-trans"

    def _generate_sign(self, text, salt, curtime):
        """
        生成签名

        :param text: 待翻译文本
        :param salt: 随机值
        :param curtime: 时间戳
        :return: 签名
        """
        # 根据规则生成input
        if len(text) <= 20:
            input_text = text
        else:
            input_text = text[:10] + str(len(text)) + text[-10:]

        # 生成签名: sha256(appKey+input+salt+curtime+应用密钥)
        sign_str = self.app_key + input_text + salt + curtime + self.app_secret
        sign = hashlib.sha256(sign_str.encode("utf-8")).hexdigest()
        return sign

    def translate_stream(
        self, text, from_lang="auto", to_lang="en", stream_type="full", prompt=None
    ):
        """
        流式翻译文本

        :param text: 待翻译文本
        :param from_lang: 源语言 (默认: auto)
        :param to_lang: 目标语言 (默认: en)
        :param stream_type: 流式返回类型 (increment/full/all)
        :param prompt: 提示词 (可选)
        :return: 流式响应迭代器
        """
        # 生成必要参数
        salt = str(uuid.uuid4())
        curtime = str(int(time.time()))

        # 生成签名
        sign = self._generate_sign(text, salt, curtime)

        # 构造请求参数
        data = {
            "appKey": self.app_key,
            "salt": salt,
            "signType": "v3",
            "sign": sign,
            "curtime": curtime,
            "i": text,
            "from": from_lang,
            "to": to_lang,
            "streamType": stream_type,
        }

        # 添加可选参数
        if prompt:
            data["prompt"] = prompt

        # 发送POST请求，接收流式响应
        response = requests.post(
            self.api_url,
            data=data,
            headers={"Content-Type": "application/x-www-form-urlencoded"},
            stream=True,
        )

        # 检查响应状态
        if response.status_code != 200:
            raise Exception(f"请求失败，状态码: {response.status_code}")

        # 按行处理流式响应
        for line in response.iter_lines(decode_unicode=True):
            if line:
                try:
                    yield json.loads(line)
                except json.JSONDecodeError:
                    # 如果不是JSON格式，直接返回原始行
                    yield line


def main():
    parser = argparse.ArgumentParser(
        description="大模型翻译demo",
        epilog="示例: python3 llmtrans-demo.py -appKey online -appSecret vidu -text '你好，世界' -from zh-CHS -to en",
    )
    parser.add_argument(
        "--appKey",
        type=str,  # 参数类型
        required=True,
        help="应用ID",
    )
    parser.add_argument(
        "--appSecret",
        type=str,  # 参数类型
        required=True,
        help="应用ID密码",
    )

    parser.add_argument(
        "--text",
        type=str,  # 参数类型
        default="你好，很高兴认识你！",
        help="要翻译的文本",
    )

    parser.add_argument(
        "--fromLang",
        type=str,  # 参数类型
        default="zh-CHS",
        help="源语种",
    )

    parser.add_argument(
        "--toLang",
        type=str,  # 参数类型
        default="en",
        help="目标语种",
    )

    parser.add_argument(
        "--streamType",
        type=str,  # 参数类型
        default="increment",
        help="返回类型，increment 或 full 或 all",
    )

    args = parser.parse_args()

    # 创建翻译客户端
    translator = YouDaoLLMTranslator(args.appKey, args.appSecret)

    # 要翻译的文本
    text = args.text

    print(f"原文: {text}")
    print("翻译结果:")

    try:
        # 执行流式翻译
        for result in translator.translate_stream(
            text=text,
            from_lang=args.fromLang,
            to_lang=args.toLang,
            stream_type=args.streamType,
        ):
            if result.startswith("event:"):
                # skip event type
                continue
            if result.startswith("data:"):
                json_part = result[5:]  # 移除 'data: ' 前缀
                json_result = json.loads(json_part)
                if json_result.get("successful", False):
                    # 处理成功响应
                    data = json_result.get("data", {})
                    trans_full = data.get("transFull", "")
                    trans_incre = data.get("transIncre", "")

                    if trans_full:
                        print(f"[全量] {trans_full}")
                    if trans_incre:
                        print(f"[增量] {trans_incre}")
                else:
                    # 处理错误响应
                    code = json_result.get("code", "")
                    message = json_result.get("message", "")
                    print(f"\n错误 [{code}]: {message}")

    except Exception as e:
        print(f"翻译过程中出现错误: {e}")


if __name__ == "__main__":
    main()
