package com.youdao.aicloud.translate.utils;

import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.sse.EventSource;
import okhttp3.sse.EventSourceListener;
import okhttp3.sse.EventSources;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Objects;

public class HttpUtil {

    private static OkHttpClient httpClient = new OkHttpClient.Builder().build();

    public static byte[] doGet(String url, Map<String, String[]> header, Map<String, String[]> params, String expectContentType) {
        Request.Builder builder = new Request.Builder();
        addHeader(builder, header);
        addUrlParam(builder, url, params);
        return requestExec(builder.build(), expectContentType);
    }

    public static byte[] doPost(String url, Map<String, String[]> header, Map<String, String[]> body, String expectContentType) {
        Request.Builder builder = new Request.Builder().url(url);
        addHeader(builder, header);
        addBodyParam(builder, body, "POST");
        return requestExec(builder.build(), expectContentType);
    }

    public static byte[] doPost(String url, Map<String, String[]> header, String params, String expectContentType) {
        Request.Builder builder = new Request.Builder().url(url);
        addHeader(builder, header);
        addBodyParam(builder, params, "POST");
        return requestExec(builder.build(), expectContentType);
    }

    public static void doPost(String url, Map<String, String[]> header, Map<String, String[]> body, EventSourceListener listener) {
        Request.Builder builder = new Request.Builder().url(url);
        addHeader(builder, header);
        addBodyParam(builder, body, "POST");
        requestExec(builder.build(), listener);
    }

    private static void addHeader(Request.Builder builder, Map<String, String[]> header) {
        if (header == null) {
            return;
        }
        for (String key : header.keySet()) {
            String[] values = header.get(key);
            if (values != null) {
                for (String value : values) {
                    builder.addHeader(key, value);
                }
            }
        }
    }

    private static void addUrlParam(Request.Builder builder, String url, Map<String, String[]> params) {
        if (params == null) {
            return;
        }
        HttpUrl.Builder urlBuilder = HttpUrl.parse(url).newBuilder();
        for (String key : params.keySet()) {
            String[] values = params.get(key);
            if (values != null) {
                for (String value : values) {
                    urlBuilder.addQueryParameter(key, value);
                }
            }
        }
        builder.url(urlBuilder.build());
    }

    private static void addBodyParam(Request.Builder builder, Map<String, String[]> body, String method) {
        if (body == null) {
            return;
        }
        FormBody.Builder formBodyBuilder = new FormBody.Builder(StandardCharsets.UTF_8);
        for (String key : body.keySet()) {
            String[] values = body.get(key);
            if (values != null) {
                for (String value : values) {
                    formBodyBuilder.add(key, value);
                }
            }
        }
        builder.method(method, formBodyBuilder.build());
    }

    private static void addBodyParam(Request.Builder builder, String params, String method) {
        if (method == null) {
            return;
        }
        MediaType JSON = MediaType.parse("application/json;charset=utf-8");
        RequestBody requestBody = RequestBody.create(JSON, params);
        builder.method(method, requestBody);
    }

    private static void requestExec(Request request, EventSourceListener listener) {
        OkHttpClient client = new OkHttpClient().newBuilder().build();
        EventSource.Factory factory = EventSources.createFactory(client);
        factory.newEventSource(request, listener);
    }

    private static byte[] requestExec(Request request, String expectContentType) {
        Objects.requireNonNull(request, "okHttp request is null");

        try (Response response = httpClient.newCall(request).execute()) {
            if (response.code() == 200) {
                ResponseBody body = response.body();
                if (body != null) {
                    String contentType = response.header("Content-Type");
                    if (contentType != null && !contentType.contains(expectContentType)) {
                        String res = new String(body.bytes(), StandardCharsets.UTF_8);
                        System.out.println(res);
                        return null;
                    }
                    return body.bytes();
                }
                System.out.println("response body is null");
            } else {
                System.out.println("request failed, http code: " + response.code());
                return response.body() == null ? null : response.body().bytes();
            }
        } catch (IOException ioException) {
            System.out.println("request exec error: " + ioException.getMessage());
        }
        return null;
    }

    public static byte[] doPost(String url, Map<String, String[]> header, Map<String, String[]> body, String parameterName, File file, String expectContentType) {
        MultipartBody.Builder builder = new MultipartBody.Builder().setType(MultipartBody.FORM);
        addFile(builder, parameterName, file);
        addBodyParam(builder, body);
        Request request = new Request.Builder().url(url).post(builder.build()).build();
        return requestExec(request, expectContentType);
    }

    /**
     * 发送普通表单数据作为 multipart/form-data (不包含文件)
     *
     * @param url               请求地址
     * @param header            请求头
     * @param body              表单数据
     * @param expectContentType 期望的内容类型
     * @return 响应字节数据
     */
    public static byte[] doPostFormData(String url, Map<String, String[]> header, Map<String, String[]> body, String expectContentType) {
        MultipartBody.Builder builder = new MultipartBody.Builder().setType(MultipartBody.FORM);
        addBodyParam(builder, body);
        Request request = new Request.Builder().url(url).post(builder.build()).build();
        return requestExec(request, expectContentType);
    }

    private static void addFile(MultipartBody.Builder builder, String parameterName, File file) {
        builder.addFormDataPart(parameterName, file.getName(), RequestBody.create(MediaType.parse("multipart/form-data"), file));
    }

    private static void addBodyParam(MultipartBody.Builder builder, Map<String, String[]> body) {
        if (body == null) {
            return;
        }
        for (String key : body.keySet()) {
            String[] values = body.get(key);
            if (values != null) {
                for (String value : values) {
                    builder.addFormDataPart(key, value);
                }
            }
        }
    }
}
